/*
 * Decompiled with CFR 0.152.
 */
package com.unboundid.common.update;

import com.unboundid.common.manifest.FileRecord;
import com.unboundid.common.manifest.Manifest;
import com.unboundid.common.manifest.ManifestHelper;
import com.unboundid.common.types.FileException;
import com.unboundid.common.update.FileActionLogger;
import com.unboundid.common.update.FileManager;
import com.unboundid.common.update.FileUpdatePolicy;
import com.unboundid.common.util.FileDigest;
import com.unboundid.common.util.FileUtils;
import com.unboundid.common.util.cli.CLIException;
import com.unboundid.common.util.cli.ConsoleInteraction;
import com.unboundid.common.util.cli.MenuBuilder;
import com.unboundid.common.util.cli.MenuCallback;
import com.unboundid.common.util.cli.MenuResult;
import com.unboundid.messages.Message;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.logging.Level;
import java.util.logging.Logger;

public class FileUpdater {
    private static final Logger LOG = Logger.getLogger(FileUpdater.class.getName());
    private ConsoleInteraction consoleInteraction;
    private File oldRoot;
    private File newRoot;
    private File backupDir;
    private FileDigest digester = FileDigest.usingMd5();
    private ManifestHelper manifest;
    private boolean isInteractive;
    private boolean ignoreWarnings;
    private Boolean replaceAll;
    private Boolean replaceNone;
    private FileUpdatePolicy updatePolicy;
    private FileManager fileManager;
    private FileActionLogger fileLog;

    public FileUpdater(File oldRoot, File newRoot, File backupDir, Manifest manifest, FileActionLogger fileLog, ConsoleInteraction consoleInteraction, boolean isInteractive, boolean ignoreWarnings, FileUpdatePolicy updatePolicy) throws IOException {
        this.oldRoot = oldRoot;
        this.newRoot = newRoot;
        this.backupDir = backupDir;
        this.manifest = new ManifestHelper(manifest, oldRoot);
        this.fileLog = fileLog;
        this.consoleInteraction = consoleInteraction;
        this.isInteractive = isInteractive;
        this.ignoreWarnings = ignoreWarnings;
        this.updatePolicy = updatePolicy;
        this.fileManager = new FileManager();
        this.fileManager.setFileDigest(FileDigest.usingMd5());
    }

    public boolean update() throws FileException {
        boolean replacedFiles = this.handleCurrentFile(this.oldRoot);
        boolean newFiles = this.copyNewFiles(this.newRoot);
        return replacedFiles || newFiles;
    }

    private boolean handleCurrentFile(File file) throws FileException {
        boolean fileSystemMod = false;
        FileUpdatePolicy.Directive directive = this.updatePolicy.getDirective(file);
        if (file.isDirectory()) {
            File[] children = file.listFiles();
            if (children != null) {
                for (File child : children) {
                    if (!this.handleCurrentFile(child)) continue;
                    fileSystemMod = true;
                }
            }
            if (FileUpdatePolicy.Directive.DELETE.equals((Object)directive)) {
                this.deleteFile(file);
            }
        } else {
            if (FileUpdatePolicy.Directive.IGNORE.equals((Object)directive)) {
                LOG.log(Level.INFO, "Ignoring " + file.getPath());
                return false;
            }
            if (FileUpdatePolicy.Directive.DELETE.equals((Object)directive)) {
                LOG.log(Level.INFO, "Deleting " + file.getPath());
                this.deleteFile(file);
                return true;
            }
            if (FileUpdatePolicy.Directive.BACKUP.equals((Object)directive)) {
                LOG.log(Level.INFO, "Backing up " + file.getPath());
                this.backupFile(file);
                return false;
            }
            boolean replace = false;
            boolean delete = false;
            String relativePath = FileUtils.getRelativePath(file, this.oldRoot);
            File srcFile = new File(this.newRoot, relativePath);
            Boolean fileModSinceDeployment = null;
            try {
                fileModSinceDeployment = this.fileModifiedSinceDeployment(file);
            }
            catch (IOException ioe) {
                LOG.log(Level.WARNING, "Error determining check sum for file " + file.getPath());
            }
            FileRecord fileRecord = this.manifest.getFileRecord(file);
            if (FileUpdatePolicy.Directive.IGNORE_IF_MODIFIED.equals((Object)directive) && Boolean.TRUE.equals(fileModSinceDeployment)) {
                LOG.log(Level.INFO, "File " + file.getPath() + " has been modified and will be ignored.");
                replace = false;
                delete = false;
            } else if (FileUpdatePolicy.Directive.REPLACE.equals((Object)directive)) {
                if (srcFile.exists()) {
                    replace = true;
                    delete = false;
                } else if (fileRecord != null) {
                    LOG.log(Level.INFO, "File " + file + " targeted for replacement but has no source file so will be deleted instead");
                    replace = false;
                    delete = true;
                } else {
                    LOG.log(Level.INFO, "File " + file + " targeted for replacement but has no source file and does not appear in the original manifest and so will be ignored instead");
                }
            } else if (fileRecord != null) {
                if (srcFile.exists()) {
                    replace = this.determineReplacement(file, fileRecord.getChecksum(), srcFile, directive, fileModSinceDeployment);
                } else {
                    LOG.log(Level.INFO, "File " + file + " has no newer version in the upgrading package.  Since this file is no longer shipped with the product it will be removed.");
                    delete = true;
                }
            }
            if (replace) {
                if (srcFile.exists()) {
                    if (Boolean.TRUE.equals(this.filesDiffer(file, srcFile))) {
                        fileSystemMod = this.replaceFile(file, srcFile);
                    } else {
                        LOG.log(Level.INFO, "File " + file + " is the same as its new version file and so will not be replaced.");
                    }
                } else {
                    LOG.log(Level.WARNING, "Attempt to replace file " + file.getPath() + " failed because no source file is available");
                }
            } else if (delete) {
                if (this.deleteFile(file)) {
                    fileSystemMod = true;
                } else {
                    LOG.log(Level.WARNING, "Failed to delete file " + file);
                }
            }
        }
        if (!fileSystemMod) {
            LOG.log(Level.INFO, "No action taken for " + file.getPath());
        }
        return fileSystemMod;
    }

    private Boolean fileModifiedSinceDeployment(File file) throws IOException {
        String originalCheckSum;
        FileRecord record = this.manifest.getFileRecord(file);
        if (record != null && (originalCheckSum = record.getChecksum()) != null) {
            String currentCheckSum = this.digester.hexStringForFile(file);
            return !originalCheckSum.equals(currentCheckSum);
        }
        return null;
    }

    private Boolean filesDiffer(File f1, File f2) {
        try {
            String cs1 = this.digester.hexStringForFile(f1);
            String cs2 = this.digester.hexStringForFile(f2);
            if (cs1 == null || cs1.length() == 0) {
                return null;
            }
            return !cs1.equals(cs2);
        }
        catch (Exception e) {
            LOG.log(Level.WARNING, "Error check sum for files " + f1.getPath() + " and " + f2.getPath(), e);
            return null;
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private boolean determineReplacement(File file, String checksum, File srcFile, FileUpdatePolicy.Directive directive, Boolean modified) throws FileException {
        if (FileUpdatePolicy.Directive.IGNORE.equals((Object)directive)) return false;
        if (Boolean.TRUE.equals(modified) && FileUpdatePolicy.Directive.IGNORE_IF_MODIFIED.equals((Object)directive)) {
            return false;
        }
        if (Boolean.FALSE.equals(modified)) {
            String newFileCheckSum = null;
            try {
                newFileCheckSum = this.digester.hexStringForFile(srcFile);
            }
            catch (IOException e) {
                LOG.log(Level.WARNING, "Error check summing file " + file, e);
            }
            if (newFileCheckSum != null) {
                if (!newFileCheckSum.equals(checksum)) {
                    LOG.log(Level.INFO, "File " + file + " will be replaced with a newer version");
                    return true;
                }
                LOG.log(Level.INFO, "File " + file + " has not changed with this new version of the server");
                return false;
            }
            if (!this.isConfigFile(file)) {
                LOG.log(Level.INFO, "New check sum not available but non-config file " + file + " will be replaced with the new version.");
                return true;
            }
            LOG.log(Level.INFO, "New check sum not available but config file " + file + " will not be replaced since it is a config file.");
            return false;
        }
        if (Boolean.TRUE.equals(modified)) {
            if (FileUpdatePolicy.Directive.IGNORE_IF_MODIFIED.equals((Object)directive)) {
                LOG.log(Level.INFO, "Config file " + file + " has changed and will not be replaced with a newer version");
                return false;
            }
            if (FileUpdatePolicy.Directive.REPLACE.equals((Object)directive)) {
                return true;
            }
            if (this.replaceNone != null && this.replaceNone.booleanValue()) {
                return false;
            }
            if (this.replaceAll != null && this.replaceAll.booleanValue()) {
                return true;
            }
            if (this.isInteractive) {
                try {
                    return this.askUserAboutReplace(Message.raw((CharSequence)("File " + file.getPath() + " has been modified since it was deployed.  How would you like to handle this file?"), (Object[])new Object[0]));
                }
                catch (CLIException e) {
                    throw new FileException(e.getMessageObject(), (Throwable)e);
                }
            }
            if (this.ignoreWarnings) {
                return true;
            }
            Message msg = Message.raw((CharSequence)("File " + file.getPath() + " has been modified since it was originally deployed.  To have this file be replaced with the new version rerun this tool specifying --" + "ignoreWarnings"), (Object[])new Object[0]);
            this.consoleInteraction.println(msg);
            throw new FileException(msg, null);
        }
        if (FileUpdatePolicy.Directive.IGNORE_IF_MODIFIED.equals((Object)directive) && this.isConfigFile(file)) {
            LOG.log(Level.INFO, "Original check sum not available for file" + file + " but it is a config file and will not be replaced");
            return false;
        }
        LOG.log(Level.INFO, "Original check sum not available for file" + file + ".  It will be replaced with a new version.");
        return true;
    }

    private boolean copyNewFiles(File newFile) throws FileException {
        boolean newFileCopied = false;
        String path = FileUtils.getRelativePath(newFile, this.newRoot);
        File currentFile = new File(this.oldRoot, path);
        if (newFile.isDirectory()) {
            File[] children = newFile.listFiles();
            if (children != null) {
                for (File child : children) {
                    if (!this.copyNewFiles(child)) continue;
                    newFileCopied = true;
                }
            }
            if (!(children != null && children.length != 0 || currentFile.exists())) {
                if (currentFile.mkdir()) {
                    newFileCopied = true;
                    this.fileLog.logAdd(currentFile);
                } else {
                    LOG.log(Level.WARNING, "Failed to create directory " + currentFile.getPath());
                }
            }
        } else {
            FileUpdatePolicy.Directive d = this.updatePolicy.getDirective(newFile);
            if (FileUpdatePolicy.Directive.IGNORE.equals((Object)d)) {
                return false;
            }
            if (!currentFile.exists()) {
                ArrayList<File> parentsToBeAdded = new ArrayList<File>();
                for (File parent = currentFile.getParentFile(); parent != null && !parent.exists(); parent = parent.getParentFile()) {
                    parentsToBeAdded.add(parent);
                }
                Collections.reverse(parentsToBeAdded);
                LOG.log(Level.INFO, "Copying new or replacement file " + newFile.getPath() + " to " + currentFile.getPath());
                this.fileManager.copy(newFile, currentFile.getParentFile(), false);
                for (File nonExistentParent : parentsToBeAdded) {
                    this.fileLog.logAdd(nonExistentParent);
                }
                this.fileLog.logAdd(currentFile);
                Runnable addAction = this.updatePolicy.getAddAction(newFile);
                if (addAction != null) {
                    addAction.run();
                }
                return true;
            }
        }
        return newFileCopied;
    }

    private boolean askUserAboutReplace(Message prompt) throws CLIException {
        MenuBuilder<Boolean> mb = new MenuBuilder<Boolean>(this.consoleInteraction);
        mb.setPrompt(prompt);
        mb.addNumberedOption(Message.raw((CharSequence)"Replace it with the new version of the file", (Object[])new Object[0]), MenuResult.success(true), new Message[0]);
        mb.addNumberedOption(Message.raw((CharSequence)"Keep the current version of the file", (Object[])new Object[0]), MenuResult.success(false), new Message[0]);
        mb.addNumberedOption(Message.raw((CharSequence)"Replace the current file and any other files modified since deployment", (Object[])new Object[0]), new MenuCallback<Boolean>(){

            @Override
            public MenuResult<Boolean> invoke(ConsoleInteraction app) throws CLIException {
                FileUpdater.this.replaceAll = true;
                return MenuResult.success(true);
            }
        }, new Message[0]);
        mb.addNumberedOption(Message.raw((CharSequence)"Keep the current file and all other files modified since deployment", (Object[])new Object[0]), new MenuCallback<Boolean>(){

            @Override
            public MenuResult<Boolean> invoke(ConsoleInteraction app) throws CLIException {
                FileUpdater.this.replaceNone = true;
                return MenuResult.success(false);
            }
        }, new Message[0]);
        return (Boolean)mb.toMenu().run().getValue();
    }

    private boolean isConfigFile(File file) {
        return FileUtils.isDescendant(file, new File(this.oldRoot, "config"));
    }

    private boolean replaceFile(File target, File srcFile) throws FileException {
        if (target != null && target.exists() && srcFile != null && srcFile.exists()) {
            this.backupFile(target);
            this.fileManager.copy(srcFile, target.getParentFile(), true);
            this.fileLog.logReplace(target);
            Runnable action = this.updatePolicy.getReplaceAction(target);
            if (action != null) {
                action.run();
            }
            return true;
        }
        return false;
    }

    private boolean deleteFile(File file) throws FileException {
        Runnable action;
        boolean isDirectory = file.isDirectory();
        boolean success = this.backupFile(file);
        FileUtils.recursiveDelete(file);
        if (success) {
            this.fileLog.logDelete(file, isDirectory);
        }
        if ((action = this.updatePolicy.getDeleteAction(file)) != null) {
            action.run();
        }
        return success;
    }

    private boolean backupFile(File file) throws FileException {
        String relativePath = FileUtils.getRelativePath(file, this.oldRoot);
        File destDir = new File(this.backupDir, relativePath).getParentFile();
        this.fileManager.copyRecursively(file, destDir);
        return true;
    }
}

