/*
 * Decompiled with CFR 0.152.
 */
package com.pingidentity.provisioner.cluster;

import com.pingidentity.provisioner.cluster.ClusteringException;
import com.pingidentity.provisioner.cluster.ProvisionerClusterManager;
import com.pingidentity.provisioner.cluster.ProvisionerNodeRole;
import com.pingidentity.provisioner.cluster.ProvisionerNodeState;
import com.pingidentity.provisioner.cluster.RuntimeClusteringException;
import com.pingidentity.provisioner.cluster.StopProvisionerException;
import com.pingidentity.provisioner.store.jdbc.ProvisionerNodeStateDataManager;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.transaction.TransactionException;
import org.springframework.transaction.support.TransactionTemplate;

public class TableProvisionerClusterManager
implements ProvisionerClusterManager {
    private static final int deleteBatchSize = 500;
    private static final Logger _logger = LogManager.getLogger(TableProvisionerClusterManager.class);
    private int _nodeId;
    private ProvisionerNodeRole _nodeRole = ProvisionerNodeRole.backup;
    private int _gracePeriodMills;
    private ProvisionerNodeStateDataManager _dataAccessManager;
    private TransactionTemplate _transactionTemplate;
    private final int minBeatIntervalMs;
    private Instant lastHeartbeat = Instant.EPOCH;
    private static boolean nodeAvailableForProvisioning = false;
    private boolean _stopped = false;

    public TableProvisionerClusterManager(int nodeId, int gracePeriodMills) throws ClusteringException {
        this(nodeId, gracePeriodMills, 1000);
    }

    TableProvisionerClusterManager(int nodeId, int gracePeriodMills, int minBeatInterval) throws ClusteringException {
        this._nodeId = nodeId;
        this._gracePeriodMills = gracePeriodMills;
        this.minBeatIntervalMs = minBeatInterval;
    }

    @Override
    public int getNodeId() {
        return this._nodeId;
    }

    public void setNodeId(int nodeId) {
        this._nodeId = nodeId;
    }

    public ProvisionerNodeRole getNodeRole() {
        return this._nodeRole;
    }

    public int getGracePeriodMills() {
        return this._gracePeriodMills;
    }

    public void setGracePeriodMills(int gracePeriodMills) {
        this._gracePeriodMills = gracePeriodMills;
    }

    public void setDataAccessManager(ProvisionerNodeStateDataManager dataAccessMgr) {
        this._dataAccessManager = dataAccessMgr;
    }

    public void setTransactionTemplate(TransactionTemplate transactionTemplate) {
        this._transactionTemplate = transactionTemplate;
    }

    @Override
    public synchronized void sendHeartBeat() throws ClusteringException {
        if (this._stopped) {
            throw new StopProvisionerException("Provisioning synchronization was interrupted. It will resume on the next cycle interval");
        }
        if (nodeAvailableForProvisioning) {
            Instant now = Instant.now();
            if (now.minusMillis(this.minBeatIntervalMs).isAfter(this.lastHeartbeat)) {
                _logger.debug("Sending heartbeat for node " + this._nodeId + ": " + now);
                this._dataAccessManager.updateHeartBeat(this._nodeId, this._nodeRole);
                this.lastHeartbeat = now;
            }
        } else {
            _logger.debug("This node is not available for provisioning. Skipping sending heartbeat for node " + this._nodeId);
        }
    }

    @Override
    public synchronized boolean isActiveNode() throws ClusteringException {
        if (this._stopped) {
            throw new StopProvisionerException("Provisioning synchronization was interrupted. It will resume on the next cycle interval");
        }
        if (this._transactionTemplate == null) {
            this.updateNodeState();
            return nodeAvailableForProvisioning && this.isActiveNodeInTransaction();
        }
        try {
            Boolean result = (Boolean)this._transactionTemplate.execute(status -> {
                try {
                    this.updateNodeState();
                    return nodeAvailableForProvisioning && this.isActiveNodeInTransaction();
                }
                catch (ClusteringException e) {
                    throw new RuntimeClusteringException(e);
                }
            });
            return result != null ? result : false;
        }
        catch (RuntimeClusteringException e) {
            throw e.getClusteringCause();
        }
        catch (TransactionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof RuntimeClusteringException) {
                RuntimeClusteringException clusteringCause = (RuntimeClusteringException)cause;
                throw clusteringCause.getClusteringCause();
            }
            throw new ClusteringException(e);
        }
    }

    @Override
    public void stopNode() {
        this._stopped = true;
    }

    private void updateNodeState() throws ClusteringException {
        this.removeStaleNodeStates();
        ProvisionerNodeState nodeState = this._dataAccessManager.getNodeState(this._nodeId);
        if (nodeState != null) {
            if (this.isTableEntryAbandoned(nodeState)) {
                this._dataAccessManager.updateHeartBeatConditionally(this._nodeId, nodeState.getHeartbeat());
                this._nodeRole = nodeState.getRole();
                this.lastHeartbeat = Instant.now();
                nodeAvailableForProvisioning = true;
            }
        } else {
            this.registerNode(this._nodeId);
            nodeAvailableForProvisioning = true;
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private boolean isActiveNodeInTransaction() throws ClusteringException {
        List<ProvisionerNodeState> nodeStates = this._dataAccessManager.getAllServerStates();
        ProvisionerNodeState thisNode = this.getNodeState(this._nodeId, nodeStates).orElseThrow(() -> new ClusteringException("Could not find current node " + this._nodeId));
        boolean isActive = false;
        this.checkCurrentNodeState(thisNode);
        ProvisionerNodeState activeNode = this.getActiveNode(nodeStates);
        if (activeNode == null) {
            ProvisionerNodeState activeNodeCandidate = this.findActiveNodeCandidate(nodeStates);
            if (activeNodeCandidate == null) throw new ClusteringException("Could not find an active node candidate!");
            if (this._nodeId != activeNodeCandidate.getId()) return isActive;
            _logger.info("Node " + this._nodeId + " is promoted to an active node.");
            this._dataAccessManager.setActiveNode(this._nodeId);
            this._nodeRole = ProvisionerNodeRole.active;
            return true;
        }
        if (this._nodeId == activeNode.getId()) {
            isActive = true;
            this._nodeRole = ProvisionerNodeRole.active;
            this.sendHeartBeat();
            return isActive;
        } else {
            if (this.isAlive(activeNode)) return isActive;
            ProvisionerNodeState activeNodeCandidate = this.findActiveNodeCandidate(nodeStates);
            if (activeNodeCandidate != null) {
                if (activeNodeCandidate.getId() != this._nodeId) return isActive;
                this._dataAccessManager.setActiveNode(this._nodeId);
                this._nodeRole = ProvisionerNodeRole.active;
                return true;
            }
            _logger.info("Could not find an active node candidate!");
        }
        return isActive;
    }

    private Optional<ProvisionerNodeState> getNodeState(int nodeId, List<ProvisionerNodeState> nodeStates) {
        return nodeStates.stream().filter(node -> node.getId() == nodeId).findFirst();
    }

    private void checkCurrentNodeState(ProvisionerNodeState nodeState) throws ClusteringException {
        if (nodeState.getRole() != this._nodeRole && this._nodeRole == ProvisionerNodeRole.active) {
            if (nodeState.getRole() == ProvisionerNodeRole.dead) {
                this._nodeRole = ProvisionerNodeRole.backup;
                this._dataAccessManager.setRole(this._nodeId, this._nodeRole);
                _logger.error("Active role assumed by another node, switching to backup role.");
            } else if (nodeState.getRole() == ProvisionerNodeRole.backup) {
                this._nodeRole = ProvisionerNodeRole.backup;
                this._dataAccessManager.setRole(this._nodeId, this._nodeRole);
                _logger.info("Forced to backup role.");
            }
        }
    }

    private void registerNode(int nodeId) throws ClusteringException {
        _logger.info("Registering provisioner node: " + nodeId);
        this._dataAccessManager.registerNode(nodeId);
    }

    private ProvisionerNodeState getActiveNode(List<ProvisionerNodeState> nodeStates) throws ClusteringException {
        ArrayList<ProvisionerNodeState> activeNodes = new ArrayList<ProvisionerNodeState>(1);
        for (ProvisionerNodeState st : nodeStates) {
            if (st.getRole() != ProvisionerNodeRole.active) continue;
            activeNodes.add(st);
        }
        switch (activeNodes.size()) {
            case 0: {
                return null;
            }
            case 1: {
                return (ProvisionerNodeState)activeNodes.get(0);
            }
        }
        StringBuilder msg = new StringBuilder("Multiple active nodes: ");
        for (ProvisionerNodeState nodeState : activeNodes) {
            msg.append(nodeState.getId()).append(", ");
        }
        msg.append("forcing all to backup role.");
        _logger.error(msg.toString());
        for (ProvisionerNodeState nodeState : activeNodes) {
            this._dataAccessManager.setRole(nodeState.getId(), ProvisionerNodeRole.backup);
            nodeState.setRole(ProvisionerNodeRole.backup);
        }
        return null;
    }

    private ProvisionerNodeState findActiveNodeCandidate(List<ProvisionerNodeState> nodeStates) {
        Instant now = Instant.now();
        Instant qualificationCheckPoint = now.minusMillis(this._gracePeriodMills);
        for (ProvisionerNodeState st : nodeStates) {
            if (!st.getHeartbeat().isAfter(qualificationCheckPoint) || st.getRole() != ProvisionerNodeRole.backup) continue;
            return st;
        }
        return null;
    }

    private void removeStaleNodeStates() {
        List<ProvisionerNodeState> nodeStates = this._dataAccessManager.getAllServerStates();
        List nodeIdsToDelete = nodeStates.stream().filter(this::isTableEntryAbandoned).map(ProvisionerNodeState::getId).peek(nodeId -> _logger.debug("Deleting stale node state " + nodeId + " from the node_state table.")).collect(Collectors.toList());
        for (int i = 0; i < nodeIdsToDelete.size(); i += 500) {
            List<Integer> deleteBatch = nodeIdsToDelete.subList(i, Math.min(nodeIdsToDelete.size(), i + 500));
            this._dataAccessManager.deleteNodes(deleteBatch);
        }
    }

    private boolean isTableEntryAbandoned(ProvisionerNodeState nodeState) {
        return nodeState.getRole() == ProvisionerNodeRole.dead || !this.isAlive(nodeState);
    }

    private boolean isAlive(ProvisionerNodeState nodeState) {
        Instant mustReportTime = nodeState.getHeartbeat().plusMillis(this._gracePeriodMills);
        return mustReportTime.isAfter(Instant.now());
    }
}

