#!/usr/bin/env sh
#
# Script to generate optimized memory options for PingFederate
#

set -e

if [ -z "$PF_HOME" ]; then
  dn=$(dirname "$0")
  PF_HOME=$(
    cd "$dn/.."
    pwd
  )
fi

runProperties="$PF_HOME/bin/run.properties"
if [ ! -f "$runProperties" ]; then
  echo "$runProperties does not exist."
  exit 1
fi

#Determine runtime mode from run.properties
pfRuntimeMode=$(awk -F= '/^pf.operational.mode/{print $2}' "${runProperties}")

#Get available memory
maybeAvailable=$(free | grep Mem: | awk '{ print $7 }')
if [ "$maybeAvailable" = "available" ]; then
  #If free command presents available, use it
  availableMem=$(free | grep "Mem:" | awk '{ print $7 }')
else
  #Otherwise, use free column
  availableMem=$(free | grep "Mem:" | awk '{ print $4 }')
fi

#Heap size should be at most 80% of available memory, in logical steps
#Max heap is 8GB
if [ "$availableMem" -lt 819200 ]; then
  heap=512
elif [ "$availableMem" -lt 983040 ]; then
  heap=640
elif [ "$availableMem" -lt 1146880 ]; then
  heap=768
elif [ "$availableMem" -lt 1310720 ]; then
  heap=896
elif [ "$availableMem" -lt 1638400 ]; then
  heap=1024
elif [ "$availableMem" -lt 1966080 ]; then
  heap=1280
elif [ "$availableMem" -lt 2621440 ]; then
  heap=1536
elif [ "$availableMem" -lt 3276800 ] || [ "$pfRuntimeMode" = "CLUSTERED_CONSOLE" ]; then
  heap=2048
elif [ "$availableMem" -lt 3932160 ]; then
  heap=2560
elif [ "$availableMem" -lt 4587520 ]; then
  heap=3072
elif [ "$availableMem" -lt 5242880 ]; then
  heap=3584
elif [ "$availableMem" -lt 6553600 ]; then
  heap=4096
elif [ "$availableMem" -lt 7864320 ]; then
  heap=5120
elif [ "$availableMem" -lt 9175040 ]; then
  heap=6144
elif [ "$availableMem" -lt 10485760 ]; then
  heap=7168
elif [ "$availableMem" -ge 10485760 ]; then
  heap=8192
fi

#Get current date and time
dt=$(date "+%Y-%m-%d_%H-%M-%S-%N")
Y=$(echo $dt | cut -c 1-4)
M=$(echo $dt | cut -c 6-7)
D=$(echo $dt | cut -c 9-10)
H=$(echo $dt | cut -c 12-13)
MN=$(echo $dt | cut -c 15-16)
S=$(echo $dt | cut -c 18-19)
MIL=$(echo $dt | cut -c 21-24)
backupDateTime="$Y-$M-${D}_$H-$MN-$S-$MIL"
generatedTime="$Y/$M/$D $H:$MN:$S:$MIL"

#Output Template
file=$(
  cat <<TEMPLATE_END
# Auto-generated JVM memory options for PingFederate
#
# Generated: ${generatedTime}
#
# Each non-commented and non-empty line will be added as a Java option when PingFederate is started.
# Comments can be added to this file by starting a line with the # character.
#

#Minimum heap size
-Xms${heap}m
#Maximum heap size
-Xmx${heap}m
#Use G1GC
-XX:+UseG1GC
TEMPLATE_END
)

jvmMemoryOpts="$PF_HOME/bin/jvm-memory.options"
if [ -f "$jvmMemoryOpts" ]; then
  #Make a backup before generating a new file
  cp "$jvmMemoryOpts" "$PF_HOME/bin/jvm-memory_${backupDateTime}.options"
fi

echo "$file" >"$jvmMemoryOpts"
echo
echo "Generated new jvm-memory.options."
