#!/usr/bin/env sh
### ====================================================================== ###
##                                                                          ##
##  PingFederate Bootstrap Script                                           ##
##                                                                          ##
### ====================================================================== ###
test "${VERBOSE}" = "true" && set -x

DIRNAME=$(dirname "$0")
PROGNAME=$(basename "$0")

# Use the maximum available, or set MAX_FD != -1 to use that
MAX_FD="maximum"

warn() {
  echo "${PROGNAME}: $*"
}

die() {
  warn "$@"
  exit 1
}

darwin=false
case "$(uname)" in
Darwin*)
  darwin=true
  ;;
esac

if [ -z "$RUN_CONF" ]; then
  RUN_CONF="$DIRNAME/run.conf"
fi
if [ -r "$RUN_CONF" ]; then
  # shellcheck source=/dev/null
  . "$RUN_CONF"
fi

if [ -z "$PF_HOME" ]; then
  PF_HOME=$(cd "$DIRNAME/.." && pwd)
fi
export PF_HOME

PF_HOME_ESC=$(echo "$PF_HOME" | sed 's/ /%20/g')

RUNFILE="$PF_HOME/bin/pingfederate.pid"
if [ ! -f "$RUNFILE" ]; then
  touch "$RUNFILE"
  chmod 664 "$RUNFILE"
fi

CURRENT_PID=$(cat "$RUNFILE")
if [ -n "$CURRENT_PID" ]; then
  if kill -0 "$CURRENT_PID" 2>/dev/null; then
    /bin/echo "Another PingFederate instance with pid $CURRENT_PID is already running. Exiting."
    exit 1
  fi
fi

if ulimit -H -n >/dev/null 2>&1; then
  MAX_FD_LIMIT=$(ulimit -H -n)
  if [ "$MAX_FD" = "maximum" ] || [ "$MAX_FD" = "max" ]; then
    MAX_FD="$MAX_FD_LIMIT"
  fi

  if [ "$darwin" = "false" ]; then
    if ! ulimit -n "$MAX_FD"; then
      warn "Could not set maximum file descriptor limit: $MAX_FD"
    fi
  fi
else
  warn "Could not query system maximum file descriptor limit: $MAX_FD_LIMIT"
fi

if [ -z "$JAVA" ]; then
  if [ -n "$JAVA_HOME" ]; then
    JAVA="$JAVA_HOME/bin/java"
  else
    JAVA="java"
    warn "JAVA_HOME is not set.  Unexpected results may occur."
    warn "Set JAVA_HOME to the directory of your local JDK or JRE to avoid this message."
  fi
fi

JAVA_VERSION=$("$JAVA" -version 2>&1 | grep 'version' | head -n 1 | cut -d\" -f 2)
JAVA_MAJOR_VERSION=$(echo "$JAVA_VERSION" | sed 's/_/./g' | cut -d. -f 1)

# Setup the classpath
pfrunjar="$PF_HOME/bin/pf-startup.jar"
if [ ! -f "$pfrunjar" ]; then
  die "Missing required file: $pfrunjar"
fi

xmlbeans="$PF_HOME/server/default/lib/xmlbeans.jar"
if [ ! -f "$xmlbeans" ]; then
  die "Missing required file: $xmlbeans"
fi

pfxml="$PF_HOME/server/default/lib/pf-xml.jar"
if [ ! -f "$pfxml" ]; then
  die "Missing required file: $pfxml"
fi

pfsdkxml="$PF_HOME/server/default/lib/pingfederate-sdk-xml.jar"
if [ ! -f "$pfsdkxml" ]; then
  die "Missing required file: $pfsdkxml"
fi

# XML classes need to be on the app class loader so that they are accessible
# in case Jetty's app classloader is not the TCCL.
PF_BOOT_CLASSPATH="$pfrunjar:$PF_HOME/startup/*:$xmlbeans:$pfxml:$pfsdkxml"

if [ -z "$PF_CLASSPATH" ]; then
  PF_CLASSPATH="$PF_BOOT_CLASSPATH"
else
  PF_CLASSPATH="$PF_CLASSPATH:$PF_BOOT_CLASSPATH"
fi

# If JAVA_OPTS is not set try check for Hotspot
if [ -z "$JAVA_OPTS" ]; then

  # Check for SUN(tm) JVM w/ HotSpot support
  HAS_HOTSPOT=$($JAVA -version 2>&1 | grep 'HotSpot')

  # Enable -server if we have Hotspot, unless we can't
  if [ -n "$HAS_HOTSPOT" ]; then
    # MacOS does not support -server flag
    if [ "$darwin" != "true" ]; then
      JAVA_OPTS="-server"
    fi
  fi
fi

# JVM Optimizations
# To configure JVM options please edit jvm-memory.options in
# <install-dir>/pingfederate/bin

jvmmemoryopts="$PF_HOME/bin/jvm-memory.options"

# Check for jvm-memory.options (used by PingFederate to set JVM memory settings)
if [ ! -f "$jvmmemoryopts" ]; then
  die "Missing $jvmmemoryopts"
fi

#Setup JVM Heap optimizations
JVM_MEMORY_OPTS=$(awk '/^-X/ { {printf "%s%1s\n", $0, ""}; }' "${jvmmemoryopts}" | tr -d '\n')

JAVA_OPTS="$JAVA_OPTS $JVM_MEMORY_OPTS"

# Setup PingFederate specific properties
JAVA_OPTS="$JAVA_OPTS -Dprogram.name=$PROGNAME"

# java 17 and 21 support
if [ "$JAVA_MAJOR_VERSION" -eq 17 ] || [ "$JAVA_MAJOR_VERSION" -eq 21 ]; then
  JAVA_OPTS="$JAVA_OPTS --add-opens=java.base/java.lang=ALL-UNNAMED"
  JAVA_OPTS="$JAVA_OPTS --add-opens=java.base/java.util=ALL-UNNAMED"
  JAVA_OPTS="$JAVA_OPTS --add-exports=java.base/sun.security.x509=ALL-UNNAMED"
  JAVA_OPTS="$JAVA_OPTS --add-exports=java.base/sun.security.util=ALL-UNNAMED"
  JAVA_OPTS="$JAVA_OPTS --add-exports=java.naming/com.sun.jndi.ldap=ALL-UNNAMED"
  JAVA_OPTS="$JAVA_OPTS --add-exports=java.base/sun.net.util=ALL-UNNAMED"
  JAVA_OPTS="$JAVA_OPTS --add-exports=java.base/sun.security.pkcs=ALL-UNNAMED"
  JAVA_OPTS="$JAVA_OPTS --add-exports=java.base/sun.security.pkcs10=ALL-UNNAMED"
  JAVA_OPTS="$JAVA_OPTS --add-exports=java.base/sun.security.internal.spec=ALL-UNNAMED"
fi

RANDOM_SOURCE="-Djava.security.egd=file:/dev/./urandom"
JAVA_OPTS="$JAVA_OPTS $RANDOM_SOURCE"

# Workaround for nCipher HSM to support Java 8
# Remove this when nCipher officially supports Java 8

JAVA_OPTS="$JAVA_OPTS -Dcom.ncipher.provider.announcemode=on"

# Prevent RHEL from enabling the JDK's built-in FIPS mode as PingFederate is not compatible with it.
# Administrators are expected to enable BCFIPS mode to be FIPS compliant.
JAVA_OPTS="$JAVA_OPTS -Dcom.redhat.fips=false"

# JAVA_OPTS="-Djetty51.encode.cookies=CookieName1,CookieName2 $JAVA_OPTS"

# Debugger arguments
#JAVA_OPTS="-Xdebug -Xrunjdwp:transport=dt_socket,address=8787,server=y,suspend=y $JAVA_OPTS"

# disable use of preallocated exceptions and always show the full stacktrace
JAVA_OPTS="$JAVA_OPTS -XX:-OmitStackTraceInFastThrow"

# comment out to disable java garbage collection log
GC_FILE="$PF_HOME/log/jvm-garbage-collection.log"

# set the number of garbage collection log files to use rotating logs, must be >= 1.
GC_FILE_COUNT="3"

# set the size of the garbage collection log file at which point the log will be rotated, must be >= 8K.
GC_FILE_SIZE="1M"

JAVA_PRODUCT_VERSION="$JAVA_MAJOR_VERSION"
mkdir -p "$PF_HOME/log"
if [ "$JAVA_PRODUCT_VERSION" -eq "1" ]; then
  JAVA_PRODUCT_VERSION=$(echo "$JAVA_VERSION" | sed 's/_/./g' | cut -d. -f 2)
fi
if [ "$JAVA_PRODUCT_VERSION" -le "8" ]; then
  GC_FLAG="-XX:+PrintGCDetails -Xloggc:"
  GC_OPTIONS=" -XX:+PrintGCTimeStamps -XX:+PrintGCDateStamps -XX:+UseGCLogFileRotation -XX:NumberOfGCLogFiles=$GC_FILE_COUNT -XX:GCLogFileSize=$GC_FILE_SIZE"
else
  GC_FLAG="-Xlog:gc*:file="
  GC_OPTIONS=":time,uptime,level,tags:filecount=$GC_FILE_COUNT,filesize=$GC_FILE_SIZE"
fi

# Setup the java endorsed dirs
PF_ENDORSED_DIRS="$PF_HOME/lib/endorsed"

#comment out to disable java crash logs
ERROR_FILE="-XX:ErrorFile=$PF_HOME_ESC/log/java_error%p.log"

#uncomment to enable Memory Dumps or set an environment variable ENABLE_AUTOMATIC_HEAP_DUMP=true
#HEAP_DUMP="-XX:+HeapDumpOnOutOfMemoryError -XX:HeapDumpPath=$PF_HOME_ESC/log"
if [ "$ENABLE_AUTOMATIC_HEAP_DUMP" = "true" ]; then
  HEAP_DUMP="-XX:+HeapDumpOnOutOfMemoryError -XX:HeapDumpPath=$PF_HOME_ESC/log"
fi

ENDORSED_DIRS_FLAG=""
if [ "$JAVA_MAJOR_VERSION" = "1" ]; then
  ENDORSED_DIRS_FLAG='-Djava.endorsed.dirs="$PF_ENDORSED_DIRS"'
fi

# Check for run.properties (used by PingFederate to configure ports, etc.)
runprops="$PF_HOME/bin/run.properties"
if [ ! -f "$runprops" ]; then
  warn "Missing run.properties; using defaults."
  runprops=""
fi

trap 'kill $PID; wait $PID; cat </dev/null >/dev/null >$RUNFILE' 1 2 3 6 15

STATUS=10
while [ "$STATUS" -eq 10 ]; do
  # Execute the JVM
  # shellcheck disable=SC2086
  "$JAVA" \
    $JAVA_OPTS \
    $ERROR_FILE \
    $HEAP_DUMP \
    ${GC_FILE:+$GC_FLAG"$GC_FILE"$GC_OPTIONS} \
    $ENDORSED_DIRS_FLAG \
    -Dlog4j2.AsyncQueueFullPolicy=Discard \
    -Dlog4j2.DiscardThreshold=INFO \
    -Dlog4j.configurationFile="$PF_HOME_ESC/server/default/conf/log4j2.xml" \
    -Drun.properties="$runprops" \
    -Dpf.home="$PF_HOME" \
    -Djetty.home="$PF_HOME" \
    -Djetty.base="$PF_HOME/bin" \
    -Djetty.server=com.pingidentity.appserver.jetty.PingFederateInit \
    -Dpf.server.default.dir="$PF_HOME/server/default" \
    -Dpf.java="$JAVA" \
    -Dpf.java.opts="$JAVA_OPTS -Drun.properties=$runprops" \
    -Dpf.classpath="$PF_CLASSPATH" \
    -Djava.library.path="$PF_HOME/startup" \
    -classpath "$PF_CLASSPATH" \
    org.pingidentity.RunPF "$@" &
  PID=$!
  /bin/echo "$PID" >"$RUNFILE" 2>/dev/null
  wait "$PID"
  STATUS=$?

  cat </dev/null >"$RUNFILE" 2>/dev/null
done
